#!/usr/bin/bash
#
#   bazel.sh - export build environment to bazel
#
#   Copyright (c) 2024 Pacman Development Team <pacman-dev@lists.archlinux.org>
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

[[ -n "$LIBMAKEPKG_BUILDENV_BAZEL_SH" ]] && return
LIBMAKEPKG_BUILDENV_BAZEL_SH=1

MAKEPKG_LIBRARY=${MAKEPKG_LIBRARY:-'/usr/share/makepkg'}

source "$MAKEPKG_LIBRARY/util/option.sh"
source "$MAKEPKG_LIBRARY/util/util.sh"

build_options+=('bazel')
buildenv_functions+=('buildenv_bazel')

#
# FIXME: this overwrites ~/.bazelrc, and as such is only suitable for chrooted builds
#

BAZELRC_HEADER="### THIS FILE IS GENERATED BY MAKEPKG"

bazel_mkopts() {
	local opts=() vals=() split=0 o
	while (( $# )); do
		case "$1" in
		--split) split=1; shift ;;
		--*) opts+=("$1"); shift ;;
		*) break ;;
		esac
	done

	if (( split )); then
		read -ra vals <<<"$*"
	else
		vals=("$@")
	fi

	if [[ ${vals+set} ]]; then
		for o in "${opts[@]}"; do
			printf "common ${o}=%s\n" "${vals[@]}"
		done
	fi
}


bazel_mkrc() {
	local var
	printf "%s\n" "$BAZELRC_HEADER"
	if ! check_option "buildflags" "n"; then
		bazel_mkopts --split --{,host_}conlyopt "$CFLAGS"
		bazel_mkopts --split --{,host_}cxxopt "$CXXFLAGS"
		bazel_mkopts --split --{,host_}linkopt "$LDFLAGS"
	fi
	if check_buildoption "distcc" "y"; then
		for var in "${!DISTCC_@}"; do
			bazel_mkopts --{,host_}action_env "$var"
		done
	fi
	if check_buildoption "ccache" "y"; then
		for var in "${!CCACHE_@}"; do
			bazel_mkopts --{,host_}action_env "$var"
		done
		bazel_mkopts --sandbox_writable_path "${CCACHE_DIR-"$HOME/.ccache"}"
	fi
	printf "%s\n" "$BAZELRC_HEADER"
}

buildenv_bazel() {
	local bazelrc="$HOME/.bazelrc" bazelrc_tmp

	if check_buildoption "bazel" "y"; then
		if ! [[ $HOME == /build ]]; then
			warning "$(gettext "%s option was requested but %s is not %s, ignoring option")" \
				"bazel" '$HOME' "/build"
			return 0
		fi

		if ! [[ ! -s "$bazelrc" || "$(sed -n '1p; $p' "$bazelrc")" == "$BAZELRC_HEADER"$'\n'"$BAZELRC_HEADER" ]]; then
			error "$(gettext "%s option was requested but %q already exists, ignoring option")" \
				"bazel" "$bazelrc"
			return 0
		fi

		if ! bazelrc_tmp="$(mktemp "${bazelrc}.XXX")"; then
			error "$(gettext "Failed to create temporary file for %q")" "$bazelrc"
			return 1
		fi

		bazel_mkrc >"$bazelrc_tmp"

		if ! mv -f "$bazelrc_tmp" "$bazelrc"; then
			error "$(gettext "Failed to create/replace %q")" "$bazelrc"
			rm -f "$bazelrc_tmp"
			return 1
		fi
	else
		if ! [[ $HOME == /build ]]; then
			return 0
		fi

		if ! [[ ! -s "$bazelrc" || "$(sed -n '1p; $p' "$bazelrc")" == "$BAZELRC_HEADER"$'\n'"$BAZELRC_HEADER" ]]; then
			return 0
		fi

		rm -f "$bazelrc"
	fi
}
