# SPDX-License-Identifier: LGPL-2.1
"""
@package sensors.py
Python Bindings for libsensors3

use the documentation of libsensors for the low level API.
see example.py for high level API usage.

@author: Pavel Rojtberg (http://www.rojtberg.net)
@see: https://github.com/paroj/sensors.py
@copyright: LGPLv2 (same as libsensors) <http://opensource.org/licenses/LGPL-2.1>
"""

from ctypes import *
import ctypes.util
import enum
from typing import (
    Optional,
    Self,
)

_libc = cdll.LoadLibrary(ctypes.util.find_library("c"))
# see https://github.com/paroj/sensors.py/issues/1
_libc.free.argtypes = [c_void_p]

_hdl = cdll.LoadLibrary(ctypes.util.find_library("sensors"))

version = c_char_p.in_dll(_hdl, "libsensors_version").value.decode("ascii")


class SensorsError(Exception):
    pass


class ErrorWildcards(SensorsError):
    pass


class ErrorNoEntry(SensorsError):
    pass


class ErrorAccessRead(SensorsError, OSError):
    pass


class ErrorKernel(SensorsError, OSError):
    pass


class ErrorDivZero(SensorsError, ZeroDivisionError):
    pass


class ErrorChipName(SensorsError):
    pass


class ErrorBusName(SensorsError):
    pass


class ErrorParse(SensorsError):
    pass


class ErrorAccessWrite(SensorsError, OSError):
    pass


class ErrorIO(SensorsError, IOError):
    pass


class ErrorRecursion(SensorsError):
    pass


_ERR_MAP = {
    1: ErrorWildcards,
    2: ErrorNoEntry,
    3: ErrorAccessRead,
    4: ErrorKernel,
    5: ErrorDivZero,
    6: ErrorChipName,
    7: ErrorBusName,
    8: ErrorParse,
    9: ErrorAccessWrite,
    10: ErrorIO,
    11: ErrorRecursion
}


def raise_sensor_error(errno, message=''):
    raise _ERR_MAP[abs(errno)](message)


class BusId(Structure):
    _fields_ = [
        ("type", c_short),
        ("nr", c_short),
    ]


class ChipName(Structure):
    prefix: bytes
    bus: BusId
    addr: int
    path: bytes

    _fields_ = [
        ("prefix", c_char_p),
        ("bus", BusId),
        ("addr", c_int),
        ("path", c_char_p),
    ]


class Feature(Structure):
    name: bytes
    number: int
    type: int  # TODO: somehow convert to Feature.Type transparently

    _fields_ = [
        ("name", c_char_p),
        ("number", c_int),
        ("type", c_int),
    ]

    # sensors_feature_type
    class Type(enum.IntEnum):
        IN = 0x00
        FAN = 0x01
        TEMP = 0x02
        POWER = 0x03
        ENERGY = 0x04
        CURR = 0x05
        HUMIDITY = 0x06
        MAX_MAIN = 0x7
        VID = 0x10
        INTRUSION = 0x11
        MAX_OTHER = 0x12
        BEEP_ENABLE = 0x18
        MAX = 0x19
        #UNKNOWN = ...  # TODO: need a way to get INT_MAX from ctypes


class Subfeature(Structure):
    name: bytes
    number: int
    type: int
    mapping: int
    flags: int  # TODO: convert to Subfeature.Flags

    _fields_ = [
        ("name", c_char_p),
        ("number", c_int),
        ("type", c_int),
        ("mapping", c_int),
        ("flags", c_uint),
    ]

    class Flags(enum.IntFlag):
        MODE_R = 1
        MODE_W = 2
        COMPUTE_MAPPING = 4


_hdl.sensors_get_detected_chips.restype = POINTER(ChipName)
_hdl.sensors_get_features.restype = POINTER(Feature)
_hdl.sensors_get_all_subfeatures.restype = POINTER(Subfeature)
_hdl.sensors_get_label.restype = c_void_p  # return pointer instead of str so we can free it
_hdl.sensors_get_adapter_name.restype = c_char_p  # docs do not say whether to free this or not
_hdl.sensors_strerror.restype = c_char_p


### RAW API ###


def init(cfg_file: str = None):
    file = _libc.fopen(cfg_file.encode("utf-8"), "r") if cfg_file is not None else None

    result = _hdl.sensors_init(file)
    if result != 0:
        raise_sensor_error(result, "sensors_init failed")

    if file is not None:
        _libc.fclose(file)


def cleanup():
    _hdl.sensors_cleanup()


def parse_chip_name(orig_name: str) -> ChipName:
    ret = ChipName()
    err = _hdl.sensors_parse_chip_name(orig_name.encode("utf-8"), byref(ret))

    if err < 0:
        raise_sensor_error(err, strerror(err))

    return ret


def strerror(errnum):
    return _hdl.sensors_strerror(errnum).decode("utf-8")


def free_chip_name(chip: ChipName):
    _hdl.sensors_free_chip_name(byref(chip))


def get_detected_chips(match, nr) -> tuple[ChipName, int]:
    """
    @return: (chip, next nr to query)
    """
    _nr = c_int(nr)

    if match is not None:
        match = byref(match)

    chip = _hdl.sensors_get_detected_chips(match, byref(_nr))
    chip = chip.contents if bool(chip) else None
    return chip, _nr.value


def chip_snprintf_name(chip, buffer_size=200) -> str:
    """
    @param buffer_size defaults to the size used in the sensors utility
    """
    ret = create_string_buffer(buffer_size)
    err = _hdl.sensors_snprintf_chip_name(ret, buffer_size, byref(chip))

    if err < 0:
        raise_sensor_error(err, strerror(err))

    return ret.value.decode("utf-8")


def do_chip_sets(chip: ChipName):
    """
    @attention this function was not tested
    """
    err = _hdl.sensors_do_chip_sets(byref(chip))
    if err < 0:
        raise_sensor_error(err, strerror(err))


def get_adapter_name(bus: BusId):
    return _hdl.sensors_get_adapter_name(byref(bus)).decode("utf-8")


def get_features(chip: ChipName, nr: int) -> tuple[Feature, int]:
    """
    @return: (feature, next nr to query)
    """
    _nr = c_int(nr)
    feature = _hdl.sensors_get_features(byref(chip), byref(_nr))
    feature = feature.contents if bool(feature) else None
    return feature, _nr.value


def get_label(chip: ChipName, feature: Feature) -> str:
    ptr = _hdl.sensors_get_label(byref(chip), byref(feature))
    val = cast(ptr, c_char_p).value.decode("utf-8")
    _libc.free(ptr)
    return val


def get_all_subfeatures(chip: ChipName, feature: Feature, nr: int) -> tuple[Subfeature, int]:
    """
    @return: (subfeature, next nr to query)
    """
    _nr = c_int(nr)
    subfeature = _hdl.sensors_get_all_subfeatures(byref(chip), byref(feature), byref(_nr))
    subfeature = subfeature.contents if bool(subfeature) else None
    return subfeature, _nr.value


def get_value(chip: ChipName, subfeature_nr: int) -> float:
    val = c_double()
    err = _hdl.sensors_get_value(byref(chip), subfeature_nr, byref(val))
    if err < 0:
        raise_sensor_error(err, strerror(err))
    return val.value


def set_value(chip: ChipName, subfeature_nr: int, value: float):
    """
    @attention this function was not tested
    """
    val = c_double(value)
    err = _hdl.sensors_set_value(byref(chip), subfeature_nr, byref(val))
    if err < 0:
        raise_sensor_error(err, strerror(err))


### Convenience API ###
class ChipIterator:
    match: Optional[ChipName]
    nr: int

    def __init__(self, match: Optional[str] = None):
        self.match = parse_chip_name(match) if match is not None else None
        self.nr = 0

    def __iter__(self) -> Self:
        return self

    def __next__(self) -> ChipName:
        chip, self.nr = get_detected_chips(self.match, self.nr)
        if chip is None:
            raise StopIteration
        return chip

    def __del__(self):
        if self.match is not None:
            free_chip_name(self.match)

    def next(self):  # python2 compatibility
        return self.__next__()


class FeatureIterator:
    chip: ChipName
    nr: int

    def __init__(self, chip: ChipName):
        self.chip = chip
        self.nr = 0

    def __iter__(self) -> Self:
        return self

    def __next__(self) -> Feature:
        feature, self.nr = get_features(self.chip, self.nr)
        if feature is None:
            raise StopIteration
        return feature

    def next(self):  # python2 compatibility
        return self.__next__()


class SubFeatureIterator:
    chip: ChipName
    feature: Feature
    nr: int

    def __init__(self, chip: ChipName, feature: Feature):
        self.chip = chip
        self.feature = feature
        self.nr = 0

    def __iter__(self) -> Self:
        return self

    def __next__(self) -> Subfeature:
        subfeature, self.nr = get_all_subfeatures(self.chip, self.feature, self.nr)
        if subfeature is None:
            raise StopIteration
        return subfeature

    def next(self):  # python2 compatibility
        return self.__next__()
