"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BASH_DIALECTS = void 0;
exports.analyzeFile = analyzeFile;
// any shebang
const SHEBANG_REGEXP = /^#!(.+)/;
// either /path/to/env [-S] <shell> or /path/to/<shell>
// NOTE that the path does not necessarily end with `/bin/`,
//      e.g. `#!/hint/sh` is an idiom
// NOTE that `/path/to/env /path/to/<shell>` is also technically valid
//      but we do not handle it here, same as `/path/to/env KEY=VAL... <shell>`
const SHEBANG_INTERPRETER_REGEXP = /^[/](?:[^ /]+[/])*(?:env +(?:-S +)?)?([^ /]+)/;
// any empty line, or a comment on an empty line, or a shellcheck directive with
// a `shell=...` entry in it
// (the first continuous run of such lines in a file is the region eligible for
//  a `shell=...` directive)
const SHELLCHECK_SHELL_OR_EMPTY_REGEXP = /^\s*(?:#\s*shellcheck\s+(?:\S+\s+)*shell=(\w+)|#|$)/;
// List of shell dialects that we support for parsing and semantic analysis
exports.BASH_DIALECTS = [
    'sh',
    'bash',
    'dash',
    'ksh',
    'zsh',
    'csh',
    'ash',
    'busybox',
];
function getShebang(fileContent) {
    const match = SHEBANG_REGEXP.exec(fileContent);
    if (!match || !match[1]) {
        return null;
    }
    return match[1].trim();
}
function parseShebang(fileContent) {
    const shebang = getShebang(fileContent);
    if (!shebang) {
        return null;
    }
    const match = SHEBANG_INTERPRETER_REGEXP.exec(shebang);
    if (!match || !match[1]) {
        return null;
    }
    return match[1].trim();
}
function parseShellDirective(fileContent) {
    const contentLines = fileContent.split('\n');
    for (const line of contentLines) {
        const match = SHELLCHECK_SHELL_OR_EMPTY_REGEXP.exec(line);
        // stop if we have a non-empty non-comment line
        if (match === null) {
            break;
        }
        // stop if we have a `shell=...` directive
        if (match[1]) {
            return match[1].trim();
        }
        // otherwise continue to scan
    }
    return null;
}
function parseUri(uri) {
    if (uri.endsWith('.zsh')) {
        return 'zsh';
    }
    return null;
}
function analyzeFile(uri, fileContent) {
    var _a;
    const directive = parseShellDirective(fileContent);
    const shebang = parseShebang(fileContent);
    const parsed = (_a = directive !== null && directive !== void 0 ? directive : shebang) !== null && _a !== void 0 ? _a : parseUri(uri);
    const dialect = parsed !== null && parsed !== void 0 ? parsed : 'bash';
    return {
        shebang,
        directive,
        dialect: exports.BASH_DIALECTS.includes(dialect) ? dialect : null,
        isDetected: parsed !== null,
    };
}
//# sourceMappingURL=shebang.js.map